# 通知設計書 7-Not Found通知

## 概要

本ドキュメントは、Horse Webフレームワークにおける Not Found通知機能の設計を記述する。リクエストされたパスが見つからない場合にクライアントへHTTP 404エラーを返却する通知機能である。

### 本通知の処理概要

本通知は、HTTPリクエストのパスに対応するルートが登録されていない場合に、クライアントに対してHTTP 404 Not Foundレスポンスを返却する機能を提供する。

**業務上の目的・背景**：Web APIやWebアプリケーションにおいて、存在しないリソースへのアクセスがあった場合、クライアントに適切なエラーレスポンスを返すことはHTTP仕様に準拠した基本要件である。この通知により、クライアントアプリケーションはリソースが存在しないことを認識でき、URLの誤りやリンク切れを検出できる。開発者やユーザーへの明確なフィードバックにより問題の特定が容易になる。

**通知の送信タイミング**：ルーティング処理において、リクエストパスに一致するルートが見つからない場合、またはルートは存在するがどのHTTPメソッドのコールバックも登録されていない場合に送信される。複数の箇所から送信される可能性がある。

**通知の受信者**：HTTPリクエストを送信したクライアント（Webブラウザ、モバイルアプリ、他のサービス等）。

**通知内容の概要**：レスポンスボディに「Not Found」というメッセージが含まれ、HTTPステータスコード404が返却される。

**期待されるアクション**：クライアントは存在しないリソースへのアクセスを検知し、URLを確認する、別のリソースを参照する、またはユーザーに適切なエラーメッセージを表示する。

## 通知種別

HTTPレスポンス - HTTP 404 Not Found

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（HTTPレスポンス） |
| 優先度 | 即時 |
| リトライ | 無し（クライアント側で判断） |

### 送信先決定ロジック

HTTPリクエストを送信したクライアントに対してレスポンスとして返却される。送信先は自動的にリクエスト元に決定される。

## 通知テンプレート

### HTTPレスポンスの場合

| 項目 | 内容 |
|-----|------|
| HTTPステータスコード | 404 |
| ステータスメッセージ | Not Found |
| Content-Type | text/html (デフォルト) |
| 形式 | テキスト |

### 本文テンプレート

```
Not Found
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | HTTPレスポンスボディのみ |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| なし | - | 固定メッセージのため変数なし | - |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ルーティング失敗1 | TNextCaller.Nextメソッド | FFound^ = False のまま処理終了 | パスが見つからない |
| ルーティング失敗2 | TNextCaller.Nextメソッド | FCallBack.Count = 0 | ルートはあるがコールバックなし |
| ワイルドカードフォールバック | THorseRouterTree.Execute | LQueueNotFound処理で405後 | /*ルートで405だった場合の置換 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ルートが見つかった | パスとメソッドが一致するルートが存在する場合 |
| 405返却時 | メソッド不一致で405が返された場合（ただしワイルドカードフォールバック時を除く） |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[ルーティング ExecuteInternal]
    B --> C{ルート見つかった?}
    C -->|見つかった| D[コールバック実行]
    C -->|見つからない| E{FFound = False?}
    E -->|Yes| F[404 Not Found送信]
    D --> G{FCallBack.Count?}
    G -->|> 0| H{メソッドのコールバック?}
    G -->|= 0| F
    H -->|あり| I[正常処理]
    H -->|なし| J[405 Method Not Allowed]
    B --> K{Execute: Result = False?}
    K -->|Yes| L[/* ルートで再試行]
    L --> M{Result = True かつ Status = 405?}
    M -->|Yes| N[404に置換]
    M -->|No| O[そのまま]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| なし | - | 本通知はデータベースを使用しない |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| なし | - | 本通知はデータベースを更新しない |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| パス不存在 | リクエストパスに一致するルートなし | 404レスポンスを返却 |
| コールバック未登録 | ルートはあるがコールバックなし | 404レスポンスを返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | サーバー側では行わない（クライアント側で判断） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

HTTPリクエスト処理中に即座に送信される。時間帯の制限はない。

## セキュリティ考慮事項

- 固定メッセージ「Not Found」のみが返却されるため、内部情報の漏洩リスクは低い
- リクエストされたパスの詳細は返却されないため、ディレクトリ構造の推測は困難
- 大量の404リクエストはリソースの列挙攻撃（enumeration attack）の可能性があるため、監視が推奨される

## 備考

- 404レスポンスは複数箇所から送信される可能性がある
  1. `Horse.Core.RouterTree.NextCaller.pas` 行124 - FCallBack.Count = 0 の場合
  2. `Horse.Core.RouterTree.NextCaller.pas` 行130 - FFound = False の場合
  3. `Horse.Core.RouterTree.pas` 行158 - ワイルドカードルートで405だった場合の置換
- THTTPStatus.NotFoundはHorse.Commons.pasで定義されている（値: 404）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

HTTPステータスコードとルーティングフラグの関係を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatus列挙型、特にNotFound = 404 |
| 1-2 | Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | FFound: ^Boolean（行38）の役割を確認 |

**読解のコツ**: FFoundはポインタ型で、ルーティングが成功したかどうかを呼び出し元に伝達するために使用される。

#### Step 2: エントリーポイントを理解する

404送信の複数の発生箇所を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | Nextメソッド内の2箇所（行124、行130） |
| 2-2 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | Executeメソッド内（行158） |

**主要処理フロー（NextCaller.pas）**:
1. **行123-124**: FCallBack.Count = 0 の場合
   - `FResponse.Send('Not Found').Status(THTTPStatus.NotFound)`
2. **行129-130**: FFound^ = False の場合（ルート未発見）
   - `if not FFound^ then FResponse.Send('Not Found').Status(THTTPStatus.NotFound)`

**主要処理フロー（RouterTree.pas）**:
1. **行152-162**: ルーティング失敗時のフォールバック処理
   - `/*`パターンで再試行
   - 405が返された場合は404に置換（行157-158）

#### Step 3: フォールバック処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | Executeメソッド（行139-166）のワイルドカードフォールバック |

**ワイルドカードフォールバックの詳細**:
```pascal
if not Result then
begin
  LQueueNotFound := GetQueuePath('/*', False);
  try
    Result := ExecuteInternal(LQueueNotFound, LMethodType, ARequest, AResponse);
    if Result and (AResponse.Status = THTTPStatus.MethodNotAllowed.ToInteger) then
      AResponse.Send('Not Found').Status(THTTPStatus.NotFound);  // 405を404に置換
  finally
    LQueueNotFound.Free;
  end;
end;
```

### プログラム呼び出し階層図

```
THorseRouterTree.Execute (行139)
    │
    ├─ ExecuteInternal (行151) - 通常ルーティング
    │      │
    │      └─ TNextCaller.Next (行80)
    │             │
    │             ├─ [FCallBack.Count = 0]
    │             │      └─ FResponse.Send('Not Found')
    │             │             .Status(THTTPStatus.NotFound)
    │             │             (行124)
    │             │
    │             └─ [FFound^ = False]
    │                    └─ FResponse.Send('Not Found')
    │                           .Status(THTTPStatus.NotFound)
    │                           (行130)
    │
    └─ [Result = False の場合] フォールバック (行152-162)
           │
           ├─ GetQueuePath('/*', False) (行154)
           │
           └─ ExecuteInternal (行156)
                  │
                  └─ [Result = True かつ Status = 405]
                         │
                         └─ AResponse.Send('Not Found')
                                .Status(THTTPStatus.NotFound)
                                (行158) - 405を404に置換
```

### データフロー図

```
[入力]                      [処理]                           [出力]

HTTPリクエスト     ───▶ ルーティング検索               ───▶ HTTPレスポンス
GET /unknown              │                                  Status: 404
                          ▼                                  Body: "Not Found"
                    FRoute辞書検索
                          │
                          ▼
                    [パス一致なし]
                          │
                          ├─▶ FFound^ := False のまま
                          │          │
                          │          ▼
                          │   FResponse.Send('Not Found')
                          │          │
                          │          ▼
                          │   .Status(THTTPStatus.NotFound)
                          │
                          └─▶ [ワイルドカードフォールバック]
                                     │
                                     ▼
                               ExecuteInternal('/*')
                                     │
                                     ▼
                               [405が返された場合]
                                     │
                                     ▼
                               404に置換
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | ソース | ルーティング判定と404エラー通知（2箇所） |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | ルーティングツリー管理、ワイルドカードフォールバック処理 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | THTTPStatus列挙型、NotFound = 404 |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponse.Send/Statusメソッド |
